/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.http.media;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.HelidonServiceLoader;
import io.helidon.common.config.Config;
import io.helidon.http.media.spi.MediaSupportProvider;

/**
 * Interface generated from definition. Please add javadoc to the definition interface.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.http.media.MediaContextConfigBlueprint")
public interface MediaContextConfig extends MediaContextConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static MediaContextConfig.Builder builder() {
        return new MediaContextConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static MediaContextConfig.Builder builder(MediaContextConfig instance) {
        return MediaContextConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static MediaContextConfig create(Config config) {
        return MediaContextConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static MediaContextConfig create() {
        return MediaContextConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link MediaContext}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends MediaContextConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends MediaContextConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<MediaSupport> mediaSupports = new ArrayList<>();
        private boolean isMediaSupportsMutated;
        private boolean mediaSupportsDiscoverServices = true;
        private boolean registerDefaults = true;
        private Config config;
        private MediaContext fallback;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(MediaContextConfig prototype) {
            if (!isMediaSupportsMutated) {
                mediaSupports.clear();
            }
            addMediaSupports(prototype.mediaSupports());
            mediaSupportsDiscoverServices = false;
            fallback(prototype.fallback());
            registerDefaults(prototype.registerDefaults());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(MediaContextConfig.BuilderBase<?, ?> builder) {
            if (isMediaSupportsMutated) {
                if (builder.isMediaSupportsMutated) {
                    addMediaSupports(builder.mediaSupports);
                }
            } else {
                mediaSupports.clear();
                addMediaSupports(builder.mediaSupports);
            }
            mediaSupportsDiscoverServices = builder.mediaSupportsDiscoverServices;
            builder.fallback().ifPresent(this::fallback);
            registerDefaults(builder.registerDefaults());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("fallback").map(MediaContext::create).ifPresent(this::fallback);
            config.get("register-defaults").as(Boolean.class).ifPresent(this::registerDefaults);
            return self();
        }

        /**
         * Media supports to use.
         * This instance has priority over provider(s) discovered by service loader.
         * The providers are used in order of calling this method, where the first support added is the
         * first one to be queried for readers and writers.
         *
         * @param discoverServices whether to discover implementations through service loader
         * @return updated builder instance
         * @see #mediaSupports()
         */
        public BUILDER mediaSupportsDiscoverServices(boolean discoverServices) {
            this.mediaSupportsDiscoverServices = discoverServices;
            return self();
        }

        /**
         * Media supports to use.
         * This instance has priority over provider(s) discovered by service loader.
         * The providers are used in order of calling this method, where the first support added is the
         * first one to be queried for readers and writers.
         *
         * @param mediaSupports media supports
         * @return updated builder instance
         * @see #mediaSupports()
         */
        public BUILDER mediaSupports(List<? extends MediaSupport> mediaSupports) {
            Objects.requireNonNull(mediaSupports);
            isMediaSupportsMutated = true;
            this.mediaSupports.clear();
            this.mediaSupports.addAll(mediaSupports);
            return self();
        }

        /**
         * Media supports to use.
         * This instance has priority over provider(s) discovered by service loader.
         * The providers are used in order of calling this method, where the first support added is the
         * first one to be queried for readers and writers.
         *
         * @param mediaSupports media supports
         * @return updated builder instance
         * @see #mediaSupports()
         */
        public BUILDER addMediaSupports(List<? extends MediaSupport> mediaSupports) {
            Objects.requireNonNull(mediaSupports);
            isMediaSupportsMutated = true;
            this.mediaSupports.addAll(mediaSupports);
            return self();
        }

        /**
         * Media supports to use.
         * This instance has priority over provider(s) discovered by service loader.
         * The providers are used in order of calling this method, where the first support added is the
         * first one to be queried for readers and writers.
         *
         * @param mediaSupport media supports
         * @return updated builder instance
         * @see #mediaSupports()
         */
        public BUILDER addMediaSupport(MediaSupport mediaSupport) {
            Objects.requireNonNull(mediaSupport);
            this.mediaSupports.add(mediaSupport);
            isMediaSupportsMutated = true;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #fallback()
         */
        public BUILDER clearFallback() {
            this.fallback = null;
            return self();
        }

        /**
         * Existing context to be used as a fallback for this context.
         *
         * @param fallback media context to use if supports configured on this request cannot provide a good result
         * @return updated builder instance
         * @see #fallback()
         */
        public BUILDER fallback(MediaContext fallback) {
            Objects.requireNonNull(fallback);
            this.fallback = fallback;
            return self();
        }

        /**
         * Existing context to be used as a fallback for this context.
         *
         * @param fallbackConfig media context to use if supports configured on this request cannot provide a good result
         * @return updated builder instance
         * @see #fallback()
         */
        public BUILDER fallback(MediaContextConfig fallbackConfig) {
            Objects.requireNonNull(fallbackConfig);
            this.fallback = MediaContext.create(fallbackConfig);
            return self();
        }

        /**
         * Existing context to be used as a fallback for this context.
         *
         * @param consumer media context to use if supports configured on this request cannot provide a good result
         * @return updated builder instance
         * @see #fallback()
         */
        public BUILDER fallback(Consumer<MediaContextConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = MediaContext.builder();
            consumer.accept(builder);
            this.fallback(builder.build());
            return self();
        }

        /**
         * Should we register defaults of Helidon, such as String media support.
         *
         * @param registerDefaults whether to register default media supports
         * @return updated builder instance
         * @see #registerDefaults()
         */
        public BUILDER registerDefaults(boolean registerDefaults) {
            this.registerDefaults = registerDefaults;
            return self();
        }

        /**
         * Media supports to use.
         * This instance has priority over provider(s) discovered by service loader.
         * The providers are used in order of calling this method, where the first support added is the
         * first one to be queried for readers and writers.
         *
         * @return the media supports
         */
        public List<MediaSupport> mediaSupports() {
            return mediaSupports;
        }

        /**
         * Existing context to be used as a fallback for this context.
         *
         * @return the fallback
         */
        public Optional<MediaContext> fallback() {
            return Optional.ofNullable(fallback);
        }

        /**
         * Should we register defaults of Helidon, such as String media support.
         *
         * @return the register defaults
         */
        public boolean registerDefaults() {
            return registerDefaults;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "MediaContextConfigBuilder{"
                    + "mediaSupports=" + mediaSupports + ","
                    + "fallback=" + fallback + ","
                    + "registerDefaults=" + registerDefaults
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        @SuppressWarnings("unchecked")
        protected void preBuildPrototype() {
            var config = this.config == null ? Config.empty() : this.config;
            {
                var serviceLoader = HelidonServiceLoader.create(ServiceLoader.load(MediaSupportProvider.class));
                this.addMediaSupports(discoverServices(config, "media-supports", serviceLoader, MediaSupportProvider.class, MediaSupport.class, mediaSupportsDiscoverServices, mediaSupports));
            }
            new MediaContextBuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Existing context to be used as a fallback for this context.
         *
         * @param fallback media context to use if supports configured on this request cannot provide a good result
         * @return updated builder instance
         * @see #fallback()
         */
        BUILDER fallback(Optional<? extends MediaContext> fallback) {
            Objects.requireNonNull(fallback);
            this.fallback = fallback.map(io.helidon.http.media.MediaContext.class::cast).orElse(this.fallback);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class MediaContextConfigImpl implements MediaContextConfig, Supplier<MediaContext> {

            private final boolean registerDefaults;
            private final List<MediaSupport> mediaSupports;
            private final Optional<MediaContext> fallback;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected MediaContextConfigImpl(MediaContextConfig.BuilderBase<?, ?> builder) {
                this.mediaSupports = List.copyOf(builder.mediaSupports());
                this.fallback = builder.fallback();
                this.registerDefaults = builder.registerDefaults();
            }

            @Override
            public MediaContext build() {
                return MediaContext.create(this);
            }

            @Override
            public MediaContext get() {
                return build();
            }

            @Override
            public List<MediaSupport> mediaSupports() {
                return mediaSupports;
            }

            @Override
            public Optional<MediaContext> fallback() {
                return fallback;
            }

            @Override
            public boolean registerDefaults() {
                return registerDefaults;
            }

            @Override
            public String toString() {
                return "MediaContextConfig{"
                        + "mediaSupports=" + mediaSupports + ","
                        + "fallback=" + fallback + ","
                        + "registerDefaults=" + registerDefaults
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof MediaContextConfig other)) {
                    return false;
                }
                return Objects.equals(mediaSupports, other.mediaSupports())
                    && Objects.equals(fallback, other.fallback())
                    && registerDefaults == other.registerDefaults();
            }

            @Override
            public int hashCode() {
                return Objects.hash(mediaSupports, fallback, registerDefaults);
            }

        }

    }

    /**
     * Fluent API builder for {@link MediaContext}.
     */
    class Builder extends MediaContextConfig.BuilderBase<MediaContextConfig.Builder, MediaContextConfig> implements io.helidon.common.Builder<MediaContextConfig.Builder, MediaContext> {

        private Builder() {
        }

        @Override
        public MediaContextConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new MediaContextConfigImpl(this);
        }

        @Override
        public MediaContext build() {
            return MediaContext.create(this.buildPrototype());
        }

    }

}
